/*
 *  OpenVPN -- An application to securely tunnel IP networks
 *             over a single TCP/UDP port, with support for SSL/TLS-based
 *             session authentication and key exchange,
 *             packet encryption, packet authentication, and
 *             packet compression.
 *
 *  Copyright (C) 2002-2004 James Yonan <jim@yonan.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program (see the file COPYING included with this
 *  distribution); if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef EVENT_H
#define EVENT_H

#define EVENT_READ     (1<<0)
#define EVENT_WRITE    (1<<1)

#define EVENT_METHOD_SCALABLE     (1<<0)
#define EVENT_METHOD_US_TIMEOUT   (1<<1)
#define EVENT_METHOD_FAST         (1<<2)

#ifdef WIN32

struct rw_handle {
  HANDLE read;
  HANDLE write;
};

typedef const struct rw_handle *event_t;

#define UNDEFINED_EVENT (NULL)

#else

typedef int event_t;

#define UNDEFINED_EVENT (-1)

#endif

struct event_set;
struct event_set_return;

struct event_set_functions
{
  void (*free)(struct event_set *es);
  void (*reset)(struct event_set *es);
  void (*del)(struct event_set *es, event_t event);
  void (*ctl)(struct event_set *es, event_t event, unsigned int rwflags, void *arg);

  /*
   * Return status for wait:
   * -1 on signal or error
   * 0 on timeout
   * length of event_set_return if at least 1 event is returned
   */
  int  (*wait)(struct event_set *es, const struct timeval *tv, struct event_set_return *out, int outlen);
};

struct event_set_return
{
  unsigned int rwflags;
  void *arg;
};

struct event_set
{
  struct event_set_functions func;
};

/*
 * maxevents on input:  desired max number of event_t descriptors
 *                      simultaneously set with event_ctl
 * maxevents on output: may be modified down, depending on limitations
 *                      of underlying API
 */
struct event_set *event_set_init (int *maxevents, unsigned int flags);

static inline void
event_free (struct event_set *es)
{
  (*es->func.free)(es);
}

static inline void
event_reset (struct event_set *es)
{
  (*es->func.reset)(es);
}

static inline void
event_del (struct event_set *es, event_t event)
{
  (*es->func.del)(es, event);
}

static inline void
event_ctl (struct event_set *es, event_t event, unsigned int rwflags, void *arg)
{
  (*es->func.ctl)(es, event, rwflags, arg);
}

static inline int
event_wait (struct event_set *es, const struct timeval *tv, struct event_set_return *out, int outlen)
{
  return (*es->func.wait)(es, tv, out, outlen);
}

static inline void
event_set_return_init (struct event_set_return *esr)
{
  esr->rwflags = 0;
  esr->arg = NULL;
}

#endif
